/* $Id$ */

/**
 * \file 
 * File specific comments.
 *
 * Most recent change:
 *   $Date$
 * 
 * Copyright (C) 2006, 2007, 2009, 2010 The University of Sydney, Australia
 *
 * This file is part of GPlates.
 *
 * GPlates is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, version 2, as published by
 * the Free Software Foundation.
 *
 * GPlates is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef GPLATES_PROPERTYVALUES_GMLTIMEINSTANT_H
#define GPLATES_PROPERTYVALUES_GMLTIMEINSTANT_H

#include <map>

#include "GeoTimeInstant.h"

#include "feature-visitors/PropertyValueFinder.h"
#include "model/PropertyValue.h"
#include "model/XmlAttributeName.h"
#include "model/XmlAttributeValue.h"


// Enable GPlatesFeatureVisitors::get_property_value() to work with this property value.
// First parameter is the namespace qualified property value class.
// Second parameter is the name of the feature visitor method that visits the property value.
DECLARE_PROPERTY_VALUE_FINDER(GPlatesPropertyValues::GmlTimeInstant, visit_gml_time_instant)

namespace GPlatesPropertyValues
{

	class GmlTimeInstant :
			public GPlatesModel::PropertyValue
	{

	public:

		/**
		 * A convenience typedef for GPlatesUtils::non_null_intrusive_ptr<GmlTimeInstant>.
		 */
		typedef GPlatesUtils::non_null_intrusive_ptr<GmlTimeInstant> non_null_ptr_type;

		/**
		 * A convenience typedef for GPlatesUtils::non_null_intrusive_ptr<const GmlTimeInstant>.
		 */
		typedef GPlatesUtils::non_null_intrusive_ptr<const GmlTimeInstant> non_null_ptr_to_const_type;


		//! Typedef for an XML attribute (name/value) map.
		typedef std::map<GPlatesModel::XmlAttributeName, GPlatesModel::XmlAttributeValue> xml_attribute_map_type;


		virtual
		~GmlTimeInstant()
		{  }

		static
		const non_null_ptr_type
		create(
				const GeoTimeInstant &time_position_,
				const xml_attribute_map_type &time_position_xml_attributes_ = xml_attribute_map_type())
		{
			return non_null_ptr_type(new GmlTimeInstant(time_position_, time_position_xml_attributes_));
		}

		const non_null_ptr_type
		clone() const
		{
			return GPlatesUtils::dynamic_pointer_cast<GmlTimeInstant>(clone_impl());
		}

		/**
		 * Access the GeoTimeInstant which encodes the temporal position of this GmlTimeInstant.
		 *
		 * Note that there is no accessor provided which returns a non-const
		 * GeoTimeInstant. This is intentional. To modify this GmlTimeInstant,
		 * set a new GeoTimeInstant using the method @a set_time_position()
		 */
		const GeoTimeInstant &
		get_time_position() const
		{
			return get_current_revision<Revision>().time_position;
		}

		/**
		 * Set the temporal position of this GmlTimeInstant to @a tp.
		 */
		void
		set_time_position(
				const GeoTimeInstant &tp);

		// @b FIXME:  Should this function be replaced with per-index const-access to
		// elements of the XML attribute map?  (For consistency with the non-const
		// overload...)
		const xml_attribute_map_type &
		get_time_position_xml_attributes() const
		{
			return get_current_revision<Revision>().time_position_xml_attributes;
		}

		// @b FIXME:  Should this function be replaced with per-index const-access to
		// elements of the XML attribute map, as well as per-index assignment (setter) and
		// removal operations?  This would ensure that revisioning is correctly handled...
		void
		set_time_position_xml_attributes(
				const xml_attribute_map_type &tpxa);

		/**
		 * Returns the structural type associated with this property value class.
		 */
		virtual
		StructuralType
		get_structural_type() const
		{
			return STRUCTURAL_TYPE;
		}

		/**
		 * Static access to the structural type as GmlTimeInstant::STRUCTURAL_TYPE.
		 */
		static const StructuralType STRUCTURAL_TYPE;


		/**
		 * Accept a ConstFeatureVisitor instance.
		 *
		 * See the Visitor pattern (p.331) in Gamma95 for information on the purpose of
		 * this function.
		 */
		virtual
		void
		accept_visitor(
				GPlatesModel::ConstFeatureVisitor &visitor) const
		{
			visitor.visit_gml_time_instant(*this);
		}

		/**
		 * Accept a FeatureVisitor instance.
		 *
		 * See the Visitor pattern (p.331) in Gamma95 for information on the purpose of
		 * this function.
		 */
		virtual
		void
		accept_visitor(
				GPlatesModel::FeatureVisitor &visitor)
		{
			visitor.visit_gml_time_instant(*this);
		}

		virtual
		std::ostream &
		print_to(
				std::ostream &os) const;

	protected:

		// This constructor should not be public, because we don't want to allow
		// instantiation of this type on the stack.
		GmlTimeInstant(
				const GeoTimeInstant &time_position_,
				const xml_attribute_map_type &time_position_xml_attributes_) :
			PropertyValue(Revision::non_null_ptr_type(new Revision(time_position_, time_position_xml_attributes_)))
		{  }

		//! Constructor used when cloning.
		GmlTimeInstant(
				const GmlTimeInstant &other_,
				boost::optional<GPlatesModel::RevisionContext &> context_) :
			PropertyValue(
					Revision::non_null_ptr_type(
							new Revision(other_.get_current_revision<Revision>(), context_)))
		{  }

		virtual
		const Revisionable::non_null_ptr_type
		clone_impl(
				boost::optional<GPlatesModel::RevisionContext &> context = boost::none) const
		{
			return non_null_ptr_type(new GmlTimeInstant(*this, context));
		}

	private:

		/**
		 * Property value data that is mutable/revisionable.
		 */
		struct Revision :
				public PropertyValue::Revision
		{
			Revision(
					const GeoTimeInstant &time_position_,
					const xml_attribute_map_type &time_position_xml_attributes_) :
				time_position(time_position_),
				time_position_xml_attributes(time_position_xml_attributes_)
			{  }

			//! Clone constructor.
			Revision(
					const Revision &other_,
					boost::optional<GPlatesModel::RevisionContext &> context_) :
				PropertyValue::Revision(context_),
				time_position(other_.time_position),
				time_position_xml_attributes(other_.time_position_xml_attributes)
			{  }

			virtual
			GPlatesModel::Revision::non_null_ptr_type
			clone_revision(
					boost::optional<GPlatesModel::RevisionContext &> context) const
			{
				return non_null_ptr_type(new Revision(*this, context));
			}

			virtual
			bool
			equality(
					const GPlatesModel::Revision &other) const
			{
				const Revision &other_revision = dynamic_cast<const Revision &>(other);

				return time_position == other_revision.time_position &&
						time_position_xml_attributes == other_revision.time_position_xml_attributes &&
						PropertyValue::Revision::equality(other);
			}

			GeoTimeInstant time_position;
			xml_attribute_map_type time_position_xml_attributes;
		};

	};

}

#endif  // GPLATES_PROPERTYVALUES_GMLTIMEINSTANT_H
